local exports = exports or {}
local TextAnim = TextAnim or {}
---@class TextAnim:ScriptComponent
---@field effectMaterial Material
---@field quad Mesh
---@field duration number
---@field progress number [UI(Range={0.0, 1.0}, Slider)]
---@field autoPlay boolean
---@field sharedMaterial Material
---@field rt ScreenRenderTexture
TextAnim.__index = TextAnim


local util = nil      ---@type Util

local AETools = AETools or {}     ---@class AETools
AETools.__index = AETools

function AETools:new(frameRate)
    local self = setmetatable({}, AETools)
    self.key_frame_info = {}
    self.frameRate = frameRate == nil and 16 or frameRate
    return self
end

function AETools:addKeyFrameInfo(in_val, out_val, frame, val)
    local key_frame_count = #self.key_frame_info
    if key_frame_count == 0 and frame > 0 then
        self.key_frame_info[key_frame_count + 1] = {
            ["v_in"] = in_val,
            ["v_out"] = out_val,
            ["cur_frame"] = 0,
            ["value"] = val
        }
    end

    key_frame_count = #self.key_frame_info
    self.key_frame_info[key_frame_count + 1] = {
        ["v_in"] = in_val,
        ["v_out"] = out_val,
        ["cur_frame"] = frame,
        ["value"] = val
    }
    self:_updateKeyFrameInfo()
end

function AETools._remap01(a,b,x)
    if x < a then return 0 end
    if x > b then return 1 end
    return (x-a)/(b-a)
end

function AETools._cubicBezier(p1, p2, p3, p4, t)
    return {
        p1[1]*(1.-t)*(1.-t)*(1.-t) + 3*p2[1]*(1.-t)*(1.-t)*t + 3*p3[1]*(1.-t)*t*t + p4[1]*t*t*t,
        p1[2]*(1.-t)*(1.-t)*(1.-t) + 3*p2[2]*(1.-t)*(1.-t)*t + 3*p3[2]*(1.-t)*t*t + p4[2]*t*t*t,
    }
end

function AETools:_cubicBezier01(_bezier_val, p)
    local x = self:_getBezier01X(_bezier_val, p)
    return self._cubicBezier(
        {0,0},
        {_bezier_val[1], _bezier_val[2]},
        {_bezier_val[3], _bezier_val[4]},
        {1,1},
        x
    )[2]
end

function AETools:_getBezier01X(_bezier_val, x)
    local ts = 0
    local te = 1
    -- divide and conque
    repeat
        local tm = (ts+te)*0.5
        local value = self._cubicBezier(
            {0,0},
            {_bezier_val[1], _bezier_val[2]},
            {_bezier_val[3], _bezier_val[4]},
            {1,-1},
            tm)
        if(value[1]>x) then
            te = tm
        else
            ts = tm
        end
    until(te-ts < 0.0001)

    return (te+ts)*0.5
end

function AETools._mix(a, b, x)
    return a * (1-x) + b * x
end

function AETools:_updateKeyFrameInfo()
    if self.key_frame_info and #self.key_frame_info > 0 then
        self.finish_frame_time = self.key_frame_info[#self.key_frame_info]["cur_frame"]
    end
end

function AETools._getDiff(val1, val2)
    local res = nil
    if type(val1) == "table" then
        local tmp_sum = 0
        for i = 1, #val1 do
            local tmp_v = math.abs(val1[i]-val2[i])
            tmp_sum = tmp_sum + tmp_v * tmp_v
        end
        res = math.sqrt(tmp_sum)
    else
        res = math.abs(val1-val2)
    end
    return res
end

function AETools:getCurPartVal(_progress, hard_cut)
    
    local part_id, part_progress = self:_getCurPart(_progress)

    local frame1 = self.key_frame_info[part_id-1]
    local frame2 = self.key_frame_info[part_id]

    if hard_cut == true then
        return frame1["value"]
    end

    local info1 = frame1["v_out"]
    local info2 = frame2["v_in"]

    local duration = (frame2["cur_frame"]-frame1["cur_frame"])/self.frameRate
    local diff = self._getDiff(frame1["value"], frame2["value"])

    local average = diff/duration + 0.0001

    local x1 = info1[2]/100
    local y1 = x1*info1[1]/average
    local x2 = 1-info2[2]/100
    local y2 = 1-(1-x2)*info2[1]/average

    if frame1["value"] > frame2["value"] then
        x1 = info1[2]/100
        y1 = -x1*info1[1]/average
        x2 = info2[2]/100
        y2 = 1+x2*info2[1]/average
        x2 = 1-x2
    end

    local bezier_val = {x1, y1, x2, y2}

    local progress = self:_cubicBezier01(bezier_val, part_progress)

    if type(frame1["value"]) == "number" then
        return self._mix(frame1["value"], frame2["value"], progress)
    end

    local res = {}
    for i = 1, #frame1["value"] do
        res[i] = self._mix(frame1["value"][i], frame2["value"][i], progress)
    end
    return res

end

function AETools:_getCurPart(progress)
    if progress > 0.999 then
        return #self.key_frame_info, 1
    end

    for i = 1, #self.key_frame_info do
        local info = self.key_frame_info[i]
        if progress < info["cur_frame"]/self.finish_frame_time then
            return i, self._remap01(
                self.key_frame_info[i-1]["cur_frame"]/self.finish_frame_time,
                self.key_frame_info[i]["cur_frame"]/self.finish_frame_time,
                progress
            )
        end
    end
end

function AETools:clear()
    self.key_frame_info = {}
    self:_updateKeyFrameInfo()
end

function AETools:test()
    Amaz.LOGI("lrc "..tostring(self.key_frame_info), tostring(#self.key_frame_info))
end


local util = {}     ---@class Util
local json = cjson.new()
local rootDir = nil
local record_t = {}

local function getBezierValue(controls, t)
    local ret = {}
    local xc1 = controls[1]
    local yc1 = controls[2]
    local xc2 = controls[3]
    local yc2 = controls[4]
    ret[1] = 3*xc1*(1-t)*(1-t)*t+3*xc2*(1-t)*t*t+t*t*t
    ret[2] = 3*yc1*(1-t)*(1-t)*t+3*yc2*(1-t)*t*t+t*t*t
    return ret
end

local function getBezierDerivative(controls, t)
    local ret = {}
    local xc1 = controls[1]
    local yc1 = controls[2]
    local xc2 = controls[3]
    local yc2 = controls[4]
    ret[1] = 3*xc1*(1-t)*(1-3*t)+3*xc2*(2-3*t)*t+3*t*t
    ret[2] = 3*yc1*(1-t)*(1-3*t)+3*yc2*(2-3*t)*t+3*t*t
    return ret
end

local function getBezierTfromX(controls, x)
    local ts = 0
    local te = 1
    -- divide and conque
    repeat
        local tm = (ts+te)/2
        local value = getBezierValue(controls, tm)
        if(value[1]>x) then
            te = tm
        else
            ts = tm
        end
    until(te-ts < 0.0001)

    return (te+ts)/2
end

local function changeVec2ToTable(val)
    return {val.x, val.y}
end

local function changeVec3ToTable(val)
    return {val.x, val.y, val.z}
end

local function changeVec4ToTable(val)
    return {val.x, val.y, val.z, val.w}
end

local function changeCol3ToTable(val)
    return {val.r, val.g, val.b}
end

local function changeCol4ToTable(val)
    return {val.r, val.g, val.b, val.a}
end

local function changeTable2Vec4(t)
    return Amaz.Vector4f(t[1], t[2], t[3], t[4])
end

local function changeTable2Vec3(t)
    return Amaz.Vector3f(t[1], t[2], t[3])
end

local function changeTable2Vec2(t)
    return Amaz.Vector2f(t[1], t[2])
end

local function changeTable2Col3(t)
    return Amaz.Color(t[1], t[2], t[3])
end

local function changeTable2Col4(t)
    return Amaz.Color(t[1], t[2], t[3], t[4])
end

local _typeSwitch = {
    ["vec4"] = function(v)
        return changeVec4ToTable(v)
    end,
    ["vec3"] = function(v)
        return changeVec3ToTable(v)
    end,
    ["vec2"] = function(v)
        return changeVec2ToTable(v)
    end,
    ["float"] = function(v)
        return tonumber(v)
    end,
    ["string"] = function(v)
        return tostring(v)
    end,
    ["col3"] = function(v)
        return changeCol3ToTable(v)
    end,
    ["col4"] = function(v)
        return changeCol4ToTable(v)
    end,

    -- change table to userdata
    ["_vec4"] = function(v)
        return changeTable2Vec4(v)
    end,
    ["_vec3"] = function(v)
        return changeTable2Vec3(v)
    end,
    ["_vec2"] = function(v)
        return changeTable2Vec2(v)
    end,
    ["_float"] = function(v)
        return tonumber(v)
    end,
    ["_string"] = function(v)
        return tostring(v)
    end,
    ["_col3"] = function(v)
        return changeTable2Col3(v)
    end,
    ["_col4"] = function(v)
        return changeTable2Col4(v)
    end,
}

local function typeSwitch()
    return _typeSwitch
end

local function createTableContent()
    -- Amaz.LOGI("lrc", "createTableContent")
    local t = {}
    for k,v in pairs(record_t) do
        t[k] = {}
        t[k]["type"] = v["type"]
        t[k]["val"] = v["func"](v["val"])
    end
    return t
end

function util.registerParams(_name, _data, _type)
    record_t[_name] = {
        ["type"] = _type,
        ["val"] = _data,
        ["func"] = _typeSwitch[_type]
    }
end

function util.getRegistedParams()
    return record_t
end

function util.setRegistedVal(_name, _data)
    record_t[_name]["val"] = _data
end

function util.getRootDir()
    if rootDir == nil then
        local str = debug.getinfo(2, "S").source
        rootDir = str:match("@?(.*/)")
    end
    Amaz.LOGI("lrc getRootDir 123", tostring(rootDir))
    return rootDir
end

function util.registerRootDir(path)
    rootDir = path
end

function util.bezier(controls)
    local control = controls
    if type(control) ~= "table" then
        control = changeVec4ToTable(controls)
    end
    return function (t, b, c, d)
        t = t/d
        local tvalue = getBezierTfromX(control, t)
        local value =  getBezierValue(control, tvalue)
        return b + c * value[2]
    end
end

function util.remap01(a,b,x)
    if x < a then return 0 end
    if x > b then return 1 end
    return (x-a)/(b-a)
end

function util.mix(a, b, x)
    return a * (1-x) + b * x
end

function util.CreateJsonFile(file_path)
    local t = createTableContent()
    local content = json.encode(t)
    local file = io.open(util.getRootDir()..file_path, "w+b")
    if file then
      file:write(tostring(content))
      io.close(file)
    end
end

function util.ReadFromJson(file_path)
    local file = io.input(util.getRootDir()..file_path)
    local json_data = json.decode(io.read("*a"))
    local res = {}
    for k, v in pairs(json_data) do
        local func = _typeSwitch["_"..tostring(v["type"])]
        res[k] = func(v["val"])
    end
    return res
end

function util.bezierWithParams(input_val_4, min_val, max_val, in_val, reverse)
    if type(input_val_4) == "tabke" then
        if reverse == nil then
            return util.bezier(input_val_4)(util.remap01(min_val, max_val, in_val), 0, 1, 1)
        else
            return util.bezier(input_val_4)(1-util.remap01(min_val, max_val, in_val), 0, 1, 1)
        end
    else
        if reverse == nil then
            return util.bezier(util.changeVec4ToTable(input_val_4))(util.remap01(min_val, max_val, in_val), 0, 1, 1)
        else
            return util.bezier(util.changeVec4ToTable(input_val_4))(1-util.remap01(min_val, max_val, in_val), 0, 1, 1)
        end
    end
end


function util.clamp(min, max, value)
	return math.min(math.max(value, min), max)
end

function util.test()
    Amaz.LOGI("lrc", "test123")
end

local ae_attribute = {
    ["Position"]={
		[1]={{0, 16.666666667, }, {2117.89786084904, 16.666666667, }, 1.00001490558104, 443.75, }, 
		[2]={{2127.03776609678, 16.666666667, }, {2127.03776609678, 16.666666667, }, 2.00002981116208, 373.514638773854, }, 
		[3]={{2124.88799485653, 16.666666667, }, {2124.88799485653, 16.666666667, }, 3.00004471674311, 302.673068382937, }, 
		[4]={{2084.9192648704, 16.666666667, }, {2084.9192648704, 16.666666667, }, 4.00005962232415, 232.580291898391, }, 
		[5]={{1921.66769553174, 16.666666667, }, {1921.66769553174, 16.666666667, }, 5.00007452790519, 164.389669166678, }, 
		[6]={{1264.45315243009, 16.666666667, }, {1264.45315243009, 16.666666667, }, 6.00008943348623, 105.124641418457, }, 
		[7]={{594.053782006497, 16.666666667, }, {594.053782006497, 16.666666667, }, 7.00010433906727, 80.5241291364678, }, 
		[8]={{296.07504368378, 16.666666667, }, {296.07504368378, 16.666666667, }, 8.0001192446483, 65.7237026496177, }, 
		[9]={{127.799965842789, 16.666666667, }, {127.799965842789, 16.666666667, }, 9.00013415022934, 60.8867915409326, }, 
		[10]={{197.849213004298, 16.666666667, }, {197.849213004298, 16.666666667, }, 10.0001490558104, 64.4913607613886, }, 
		[11]={{330.288607219158, 16.666666667, }, {330.288607219158, 16.666666667, }, 11.0001639613914, 73.9743261193903, }, 
		[12]={{377.659399877529, 16.666666667, }, {377.659399877529, 16.666666667, }, 12.0001788669725, 86.3979314848874, }, 
		[13]={{351.548767803479, 16.666666667, }, {351.548767803479, 16.666666667, }, 13.0001937725535, 99.0227903170114, }, 
		[14]={{273.397095218126, 16.666666667, }, {273.397095218126, 16.666666667, }, 14.0002086781345, 109.714593871416, }, 
		[15]={{168.25309543536, 16.666666667, }, {168.25309543536, 16.666666667, }, 15.0002235837156, 117.156000700562, }, 
		[16]={{67.8057430949917, 16.666666667, }, {67.8057430949917, 16.666666667, }, 16.0002384892966, 120.874071534406, }, 
		[17]={{48.763037719569, 16.666666667, }, {48.763037719569, 16.666666667, }, 17.0002533948776, 121.121696482162, }, 
		[18]={{99.163355236484, 16.666666667, }, {99.163355236484, 16.666666667, }, 18.0002683004587, 118.666643393877, }, 
		[19]={{133.296933099778, 16.666666667, }, {133.296933099778, 16.666666667, }, 19.0002832060397, 114.544633249624, }, 
		[20]={{137.436943594476, 16.666666667, }, {137.436943594476, 16.666666667, }, 20.0002981116208, 109.825652126387, }, 
		[21]={{117.888207899171, 16.666666667, }, {117.888207899171, 16.666666667, }, 21.0003130172018, 105.429053544278, }, 
		[22]={{83.4030774171956, 16.666666667, }, {83.4030774171956, 16.666666667, }, 22.0003279227828, 102.006652901579, }, 
		[23]={{43.1639655927279, 16.666666667, }, {43.1639655927279, 16.666666667, }, 23.0003428283639, 99.897299272218, }, 
		[24]={{17.8591306796647, 16.666666667, }, {17.8591306796647, 16.666666667, }, 24.0003577339449, 99.1437795105617, }, 
		[25]={{25.1354706065461, 16.666666667, }, {25.1354706065461, 16.666666667, }, 25.000372639526, 99.5547345535636, }, 
		[26]={{43.7351105601181, 16.666666667, }, {43.7351105601181, 16.666666667, }, 26.000387545107, 100.790861500973, }, 
		[27]={{51.0010522327887, 16.666666667, }, {51.0010522327887, 16.666666667, }, 27.000402450688, 102.455489443815, }, 
		[28]={{48.1412448986757, 16.666666667, }, {48.1412448986757, 16.666666667, }, 28.0004173562691, 104.173533907137, }, 
		[29]={{37.9881143039227, 16.666666667, }, {37.9881143039227, 16.666666667, }, 29.0004322618501, 105.648483579972, }, 
		[30]={{23.9199400012789, 16.666666667, }, {23.9199400012789, 16.666666667, }, 30.0004471674311, 106.693116158725, }, 
		[31]={{9.88051966129461, 16.666666667, }, {9.88051966129461, 16.666666667, }, 31.0004620730122, 107.234986553089, }, 
		[32]={{6.36671252928079, 16.666666667, }, {6.36671252928079, 16.666666667, }, 32.0004769785932, 107.301658768887, }, 
		[33]={{12.8843119151794, 16.666666667, }, {12.8843119151794, 16.666666667, }, 33.0004918841743, 106.992843535264, }, 
		[34]={{17.8358291190395, 16.666666667, }, {17.8358291190395, 16.666666667, }, 34.0005067897553, 106.44709980432, }, 
		[35]={{18.6845050914818, 16.666666667, }, {18.6845050914818, 16.666666667, }, 35.0005216953363, 105.809872510928, }, 
		[36]={{16.2460948184416, 16.666666667, }, {16.2460948184416, 16.666666667, }, 36.0005366009174, 105.207839886558, }, 
		[37]={{11.6897509143372, 16.666666667, }, {11.6897509143372, 16.666666667, }, 37.0005515064984, 104.732341474913, }, 
		[38]={{6.26236287716809, 16.666666667, }, {6.26236287716809, 16.666666667, }, 38.0005664120795, 104.432510827221, }, 
		[39]={{2.47373034985598, 16.666666667, }, {2.47373034985598, 16.666666667, }, 39.0005813176605, 104.316986866046, }, 
		[40]={{1.46391699781808, 16.666666667, }, {0, 16.666666667, }, 40.0005962232415, 104.361921889893, }, 
	}, 
	["Scale"]={
		[1]={{0, 16.666666667, }, {0, 16.666666667, }, 3.00004471674311, 3.20730222595765, }, 
		[2]={{0, 16.666666667, }, {0, 16.666666667, }, 6.00008943348623, 1.77947618193638, }, 
		[3]={{0, 81, }, {0, 16.666666667, }, 8.0001192446483, 1, }, 
		[4]={{0, 81, }, {0, 16.666666667, }, 40.0005962232415, 1, }, 
	}, 
    ["GaussianBlur"]={
		[1]={{0, 16.666666667, }, {-612.1262, 16.666666667, }, 1.00001490558104, 125.8, }, 
		[2]={{-612.1262, 16.666666667, }, {-796.065599999998, 16.666666667, }, 2.00002981116208, 105.5, }, 
		[3]={{0, 50, }, {25.1283333333333, 16.666666667, }, 3.00004471674311, 79.1, }, 
		[4]={{-594.0338, 16.666666667, }, {-150.77, 16.666666667, }, 6.00008943348623, 20, }, 
		[5]={{0, 50, }, {120.616, 16.666666667, }, 10.0001490558104, 0, }, 
		[6]={{113.0775, 16.666666667, }, {-56.53875, 16.666666667, }, 14.0002086781345, 15, }, 
		[7]={{0, 50, }, {0, 16.666666667, }, 22.0003279227828, 0, }, 
		[8]={{0, 50, }, {0, 16.666666667, }, 40.0005962232415, 0, }, 
	}, 
}


function TextAnim.new(construct, ...)
    local self = setmetatable({}, TextAnim)
    self.effectMaterial = nil
    self.curTime = 0
    self.progress = 0
    self.autoPlay = false
    self.duration = 1.0
    self.quad = nil
    self.offset = 220.6380781101
    if construct and TextAnim.constructor then TextAnim.constructor(self, ...) end
    return self
end

function TextAnim:constructor()
    self.name = "scriptComp"
end

function TextAnim:initKeyFrame() 
    for _name, info_list in pairs(ae_attribute) do
        local tool = AETools:new(40)
        for i = 1, #info_list do
            tool:addKeyFrameInfo(info_list[i][1], info_list[i][2], info_list[i][3], info_list[i][4])
        end
        self[_name] = tool
    end
end

local function getRootDir()
    local rootDir = nil
    if rootDir == nil then
        local str = debug.getinfo(2, "S").source
        rootDir = str:match("@?(.*/)")
    end
    -- Amaz.LOGI("lrc getRootDir 3", tostring(rootDir))
    return rootDir
end
function TextAnim:onStart(comp)
    self.text = comp.entity:getComponent('SDFText')
    self.camera = comp.entity.scene:findEntityBy("InfoSticker_camera_entity"):getComponent("Transform")
    if self.text == nil then
        local text = comp.entity:getComponent('Text')
        if text ~= nil then
            self.text = comp.entity:addComponent('SDFText')
            self.text:setTextWrapper(text)
        end
    end
    self.renderer = nil
	if self.text ~= nil then
		self.renderer = comp.entity:getComponent("MeshRenderer")
	else
		self.renderer = comp.entity:getComponent("Sprite2DRenderer")
	end
    self.first = true
    local rect = self.text.rect
    -- self.text.targetRTExtraSize = Amaz.Vector2f(rect.width * 3.0, rect.height * 24.0)
    self.trans = comp.entity:getComponent("Transform")
    self.parentTrans = self.trans.parent
    self.parentEntity = self.parentTrans.entity
    self.layer = tostring(self.parentEntity.name)
    -- comp.entity.layer = 1
    self.rootDir = getRootDir()
    self.rootDir = string.sub(self.rootDir,1,string.len(self.rootDir)-4) 
    -- self.effectText_prefab = Amaz.PrefabManager.loadPrefab(self.rootDir, self.rootDir .. "prefabs/pack.prefab")
    -- self.customEntity = self.effectText_prefab:instantiateToEntity(comp.entity.scene, self.rootEntity)
    -- self.blurEntity = comp.entity.scene:findEntityBy("Untitled"):getComponent("MeshRenderer").material
end

function TextAnim:setMatToSDFText()

    self.text.renderToRT = true
    local materials = Amaz.Vector()
    local InsMaterials = nil
    if self.effectMaterial then
        InsMaterials = self.effectMaterial:instantiate()
    else
        InsMaterials = self.renderer.material
    end
    materials:pushBack(InsMaterials)
    self.materials = materials
    self.renderer.materials = self.materials

    self.material = self.renderer.material
end

function TextAnim:initAnimConfig()
    self.text:forceTypeSetting()
	self.count = self.text.chars:size()
    for i = 0, self.text.chars:size() - 1 do
        local char = self.text.chars:get(i)
        if char.utf8code == "\n" or char.utf8code == " " then
            self.count = self.count - 1
        end
    end
	self.CharForward = 0.96  --zi yu zi zhi jian de jian ge, fan wei shi [0, 1]
	self.CharDuration = 1.0 / (self.count - self.CharForward * (self.count - 1.0))
	self.CharForward = self.CharForward * self.CharDuration
    self.charAnchor = {}
    self.index = {}
    self.perRowSize = {}
    for i = 0, self.text.chars:size() - 1 do
        local char = self.text.chars:get(i)
        if char.utf8code ~= "\n" then
            if self.perRowSize[char.rowth] == nil then
                self.perRowSize[char.rowth] = 1
            else
                self.perRowSize[char.rowth] = self.perRowSize[char.rowth] + 1
            end
        end
    end
    local j = 0
    for i = 0, self.text.chars:size() - 1 do 
        local char = self.text.chars:get(i)
        if char.utf8code ~= "\n" and char.utf8code ~= " " then
            local x = char.idInRow + 1
            local y = self.perRowSize[char.rowth] - x - 1 + 2
            x = math.max(math.random(-x, -1), -3)
            y = math.min(math.random(1, y), 3)
            if char.idInRow % 2 == 0 then
                self.charAnchor[j] = char.width * y * 0.75
            else
                self.charAnchor[j] = char.width * x * 0.75
            end
            -- if -x > y then
            --     self.charAnchor[j] = char.width * x * 0.75
            -- elseif -x < y then
            --     self.charAnchor[j] = char.width * y * 0.75
            -- else
            --     local a = math.random()
            --     if a > 0.5 then
            --         self.charAnchor[j] = char.width * x * 0.75
            --     else
            --         self.charAnchor[j] = char.width * y * 0.75
            --     end
            -- end 
            
            self.index[j] = i
            j = j + 1
        end
    end
    for i = 0, self.count - 1 do
        local a = math.floor(math.random(0, self.count - 1))
        local b = math.floor(math.random(0, self.count - 1))

        local temp = self.index[a]
        self.index[a] = self.index[b]
        self.index[b] = temp

        temp = self.charAnchor[a]
        self.charAnchor[a] = self.charAnchor[b]
        self.charAnchor[b] = temp
    end
end

function TextAnim:onUpdate(comp, deltaTime)
    if Amaz.Macros and Amaz.Macros.EditorSDK then
        self:seek(self.curTime)
        if self.autoPlay then
            self.curTime = self.curTime + deltaTime
            self.progress = (self.curTime % self.duration) / self.duration
        end
    else
    end


end



function TextAnim:createEntity(entity, compName)
    local radiusBlurEntity = entity.scene:createEntity("radiusBlur_"..compName..self.layer)
    local trans = radiusBlurEntity:addComponent("Transform")
    trans.parent = self.camera
    self.camera.children:pushBack(trans)
    if compName == "MeshRenderer" then
        trans.localPosition = Amaz.Vector3f(0, 0, -10)
        self.radiusBlurMR = radiusBlurEntity:addComponent("MeshRenderer")
        self.radiusBlurMR.mesh = self.quad
        local materials = Amaz.Vector()
        local InsMaterials = nil
        if self.effectMaterial then
            InsMaterials = self.effectMaterial:instantiate()
        else
            InsMaterials = self.radiusBlurMR.material
        end
        materials:pushBack(InsMaterials)
        self.materials = materials
        self.radiusBlurMR.materials = self.materials
        -- self.radiusBlurMR.sharedMaterial = self.effectMaterial:instantiate()
        self.material = self.radiusBlurMR.material
    elseif compName == "Camera" then
        self.radiusBlurCamera = radiusBlurEntity:addComponent("Camera")
        self.rt1 = Amaz.ScreenRenderTexture()
        self.radiusBlurCamera.renderTexture = self.rt1
        self.radiusBlurCamera.renderTexture.shared = false
        self.radiusBlurCamera.layerVisibleMask = Amaz.DynamicBitset.new(64,"0x0")
        self.radiusBlurCamera.layerVisibleMask = self.radiusBlurCamera.layerVisibleMask:set(self.layer, true)
        self.radiusBlurCamera.renderOrder = 0
        self.radiusBlurCamera.clearColor = Amaz.Color(0.0, 0.0, 0.0, 0.0)
        self.radiusBlurCamera.clearType = 1
        self.radiusBlurCamera.type = 1
        self.radiusBlurCamera.alwaysClear = true
    end
end

function TextAnim:seek(time)
    if self.first == true then
        self:initKeyFrame()
        --self:initAnimConfig()
        self.text.targetRTExtraSize = Amaz.Vector2f(0.0, self.text.rect.height * 3.0)
	    self.text.renderToRT = true
        self:setMatToSDFText()
        local w = Amaz.BuiltinObject:getInputTextureWidth()
        local h = Amaz.BuiltinObject:getInputTextureHeight()
        -- self.text.entity.layer = self.layer * 10

        -- if self.customEntity == nil then
        --     self.effectText_prefab = Amaz.PrefabManager.loadPrefab(self.rootDir, self.rootDir .. "prefabs/pack.prefab")
        --     self.customEntity = self.effectText_prefab:instantiateToEntity(self.text.entity.scene, self.camera.entity)
        --     self.blurMeshRenderer = self.customEntity:searchEntity("Untitled"):getComponent("MeshRenderer")
        --     self.blurTrans = self.customEntity:searchEntity("Untitled"):getComponent("Transform")
        --     -- self:createEntity(self.text.entity, "MeshRenderer")
        --     -- self:createEntity(self.text.entity, "Camera")
        --     -- self.material:setTex("u_albedo", self.rt1)
        --     local materials = Amaz.Vector()
        --     local InsMaterials = nil
        --     if self.effectMaterial then
        --         InsMaterials = self.effectMaterial:instantiate()
        --     else
        --         InsMaterials = self.blurMeshRenderer.material
        --     end
        --     materials:pushBack(InsMaterials)
        --     self.materials = materials
        --     self.blurMeshRenderer.materials = self.materials

        --     self.material = self.blurMeshRenderer.material
        --     self.camera1 = self.customEntity:searchEntity("Camera"):getComponent("Camera")
        --     self.camera1Trans = self.customEntity:searchEntity("Camera"):getComponent("Transform")
        --     local rt = Amaz.ScreenRenderTexture()
        --     rt:setShared(false)
        --     self.camera1.renderTexture = rt
        --     self.material:setTex("u_albedo", self.camera1.renderTexture)
        --     -- self.material:setFloat("u_Strength", 0)
        --     local x = self.layer * 10
        --     x = math.floor(x / 64) * 64 + 64
        --     self.camera1.layerVisibleMask = Amaz.DynamicBitset.new(x,"0x0")
        --     self.camera1.layerVisibleMask = self.camera1.layerVisibleMask:set(self.layer * 10, true)
        --     -- self:setMatToSDFText()
        --     self.camera1.renderOrder = 0
        -- end
        self.first = false
    else
    end


    if Amaz.Macros and Amaz.Macros.EditorSDK then
    else
        self.progress = time / self.duration
    end
    -- self.camera1Trans:setWorldPosition(Amaz.Vector3f(0.0, 0.0, 0.0))
    -- self.blurTrans:setWorldPosition(Amaz.Vector3f(0.0, 0.0, -20.0))
    local Position = self.Position:getCurPartVal(self.progress)
    local Position1 = self.Position:getCurPartVal(math.max(self.progress - 0.0333,0.0))
    local Scale = self.Scale:getCurPartVal(self.progress)
    Position = (Position + self.offset) / 650 * 2.0 - 1.0
    Position1 = (Position1 + self.offset) / 650 * 2.0 - 1.0
    local offset = (Position - Position1)
    self.trans.localPosition = Amaz.Vector3f(0.0, -Position, -10.0)
    self.trans.localScale = Amaz.Vector3f(1.0, Scale, 1.0)
    -- local chars = self.text.chars
    -- for i = 1, self.count do
        
    --     local s = util.clamp(0, self.CharDuration, self.progress - (i - 1) * self.CharDuration + (i - 1) * self.CharForward)
    --     local progress = util.remap01(0, self.CharDuration, s)
    --     local Rotate = self.Rotate:getCurPartVal(progress)
    --     local Scale = self.Scale:getCurPartVal(math.pow(progress, 0.8))
    --     local Position = self.Position:getCurPartVal(progress)
    --     local Alpha = self.Alpha:getCurPartVal(math.min(progress * 32.0, 1.0))
    --     local char = chars:get(self.index[i - 1])
    --     local s = Scale
    --     char.scale = Amaz.Vector3f(s, s, 1.0)
    --     local a = char.initialPosition.x + self.charAnchor[i - 1]
    --     local x = (char.initialPosition.x - a) * s + a
    --     char.position = Amaz.Vector3f(x, char.position.y, char.position.z)
    --     char.anchor = Amaz.Vector2f(self.charAnchor[i - 1] + (char.initialPosition.x - x), 0.0)
    --     char.rotate = Amaz.Vector3f(0.0, 0.0, Rotate)
    --     char.color = Amaz.Vector4f(char.color.x, char.color.y, char.color.z, Alpha)
    -- end
    local blur = self.GaussianBlur:getCurPartVal(self.progress)
    local blur1 = self.GaussianBlur:getCurPartVal(math.max(self.progress - 0.0333,0.0))
    self.material:setFloat("u_Strength", (blur * 0.0015) / math.max(math.min(self.duration, 2.0), 0.8) * 3.5)
    self.material:setFloat("u_Strength1", (blur * 0.0015) / math.max(math.min(self.duration, 2.0), 0.8) * 5.0)
    self.material:setVec2("u_TextRect", Amaz.Vector2f(self.text.rect.width, self.text.rect.height))
    local p = util.remap01(0.0, 0.05, self.progress)
    self.material:setFloat("u_Alpha", p)
    self.material:setFloat("u_TrailAlpha", math.pow(1. - self.progress, 2.0))
    self.material:setFloat("u_Offset", offset * math.pow(self.progress, 1.0))
    -- self.material:setVec2("u_Center", Amaz.Vector2f(self.parentTrans.localPosition.x * 0.5 + 0.5, self.parentTrans.localPosition.y * 0.5 + 0.5))
end


function TextAnim:onEnter()
	self.first = true
	
end
function TextAnim:resetData( ... )
	if self.text ~= nil then
    	local chars = self.text.chars 
		for i = 1, self.text.chars:size() do
			local char = chars:get(i - 1)
			if char.rowth ~= -1 then
				char.position = char.initialPosition
				char.rotate = Amaz.Vector3f(0, 0, 0)
				char.scale = Amaz.Vector3f(1, 1, 1)
				char.color = Amaz.Vector4f(1, 1, 1, 1)
			end
		end
        self.text.renderToRT = false
        self.text.chars = chars
	end

	self.trans.localPosition = Amaz.Vector3f(0, 0, 0)
	self.trans.localEulerAngle = Amaz.Vector3f(0, 0, 0)
	self.trans.localScale = Amaz.Vector3f(1, 1, 1)
    self.text.targetRTExtraSize = Amaz.Vector2f(0.0, 0.0)

    -- if self.customEntity ~= nil then
    --     self.text.entity.scene:removeEntity(self.customEntity)
    --     self.camera.children:erase(self.customEntity)
    --     self.customEntity = nil
    -- end
    -- self.text.entity.layer = 0
end

function TextAnim:setDuration(duration)
   self.duration = duration
end
function TextAnim:onLeave()
	self:resetData()
	self.first = true
end
function TextAnim:clear()
	self:resetData()
end

exports.TextAnim = TextAnim
return exports
