precision highp float;
varying highp vec2 uv0;

// Max anchor points are restricted to be 25. But for hue_vs_x curve, two mirror points are added so max 
// anchor points should be (25 + 2), and then the total points num will be (25 + 2) * 3 = 81
#define MAX_POINTS 81
uniform vec2 u_points[MAX_POINTS];
uniform int u_anchorPtsNum;

const float EPS = 0.00005;

vec4 floatToVec4(float val)
{
    float d = val < 0. ? 1. : 0.; // Store sign in w component (1.0 for negative)
    float a = floor(abs(val)) / 255.;  // Recover high bits (integer part)
    float decimalPart = fract(abs(val)) * 255.;
    float b = floor(decimalPart) / 255.;   // Recover middle bits (first fractional part)
    float c = fract(decimalPart);   // Recover low bits (remaining fractional precision)
    return vec4(a, b, c, d);
}


/* Get the point at the given index. Use loop method for gles2 support */
vec2 getPoints(int index)
{
    vec2 res = vec2(0.0);
    for (int i = 0; i < MAX_POINTS; i++) {
        if (index == i) {
            res = u_points[i];
            break;
        }
    }
    return res;
}


/**
 * Computes a point on a cubic Bézier curve defined by 4 control points.
 * 
 * @param p0 Start point (anchor)
 * @param p1 First control point
 * @param p2 Second control point
 * @param p3 End point (anchor)
 * @param t  Interpolation parameter [0, 1]
 * 
 * @return Interpolated value along the curve at parameter t
 */
float computeCubicBezier(float p0, float p1, float p2, float p3, float t)
{
    // Precompute powers of t and (1-t)
    float t2 = t * t;
    float t3 = t2 * t;
    float invT = 1. - t;
    float invT2 = invT * invT;
    float invT3 = invT2 * invT;
    // Expanded cubic Bézier formula: B(t) = (1-t)^3 * p0 + 3*(1-t)^2*t * p1 + 3*(1-t)*t^2 * p2 + t^3 * p3
    float res = invT3 * p0 + 3. * invT2 * t * p1 + 3. * invT * t2 * p2 + t3 * p3;
    return res;
}


/**
 * Finds the parameter t for a given point p on a cubic Bézier curve.
 * Uses binary search with a fixed number of iterations (20 steps).
 * 
 * @param p0 Start point (anchor)
 * @param p1 First control point
 * @param p2 Second control point
 * @param p3 End point (anchor)
 * @param p  Target point value to solve for t
 * 
 * @return Parameter t ∈ [0,1] that yields p = Bézier(p0,p1,p2,p3,t)
 * 注意：此函数的参数以p来命名，但在该应用中（调色曲线），p主要指x坐标，要求x(t)与t的关系必须是单调的，才能用二分法找到t。
 *      这里所讲的'单调'关系在调色曲线中一般是满足的，等同于y必须是x的函数，同一个x不允许有多个y与之对应。
 */
float computeInverseBezierParam(float p0, float p1, float p2, float p3, float p)
{
    // Early exit for endpoints
    if (abs(p0 - p) < EPS) {
        return 0.;
    }
    if (abs(p3 - p) < EPS) {
        return 1.;
    }

    // Binary search setup
    float startT = 0.;
    float endT = 1.;
    float halfT = 0.5;
    float halfP = computeCubicBezier(p0, p1, p2, p3, halfT);

    // Binary search loop (no more than 20 iterations)
    for (int i = 0; i < 20; i++) {
        if (abs(halfP - p) < EPS) {
            break;
        }

        if (halfP < p) {
            startT = halfT;  // Search right half
        } else {
            endT = halfT;    // Search left half
        }
        halfT = (startT + endT) * 0.5;
        halfP = computeCubicBezier(p0, p1, p2, p3, halfT);
    }
    return halfT;
}


float computeBezierY(float x)
{
    // check special case (default) when there is no anchor points
    if (u_anchorPtsNum == 0) {
        return 0.5;
    }

    /*
     * Bézier points will be arranged in the following way:
     * L0, A0, R0, L1, A1, R1, L2, A2, R2, ...
     * where Li is the left control point, Ri is the right control point, Ai is the anchor point.
     * so the index of each point is:
     * Li = i*3, Ai = i*3 + 1, Ri = i*3 + 2
     */

    // Identify the anchor point that is the immediate right neighbor of the given x
    int i = 0;
    while (i < u_anchorPtsNum) {
        vec2 anchorPt = getPoints(i * 3 + 1);   // i-th anchor point
        if (x < anchorPt.x) {
            break;
        }
        i++;
    }
    i = (i < u_anchorPtsNum - 1) ? i : (u_anchorPtsNum - 1);

    // Get the quadruplet points that are used to compute the Bézier curve
    vec2 p0 = getPoints(i * 3 - 2); // A(i-1), (i-1)-th anchor point
    vec2 p1 = getPoints(i * 3 - 1); // R(i-1), (i-1)-th right control point
    vec2 p2 = getPoints(i * 3);     // L(i), i-th left control point
    vec2 p3 = getPoints(i * 3 + 1); // A(i), i-th anchor point

    // Compute the Bézier curve at the given x using the 4 points
    float t = computeInverseBezierParam(p0.x, p1.x, p2.x, p3.x, x);
    float y = computeCubicBezier(p0.y, p1.y, p2.y, p3.y, t);
    return y;
}


void main()
{
    float y = computeBezierY(uv0.x);
    gl_FragColor = floatToVec4(y);
}
