precision highp float;
varying highp vec2 uv0;
uniform sampler2D u_inputTexture;
uniform sampler2D u_curveHue2hue;
uniform sampler2D u_curveHue2sat;
uniform sampler2D u_curveHue2lum;
uniform sampler2D u_curveLum2sat;
uniform sampler2D u_curveSat2sat;
uniform sampler2D u_curveSat2lum;

const vec3 LuminanceWeight = vec3(0.2126, 0.7152, 0.0722);  // BT.709 weight
const float StepX = 0.00390625; // 1/256

// Parameters for coordinates scale
const float HueRange = 360.;
const float HueYmin = -180.;
const float HueYmax = 180.;

const float SatYmin = 0.;
const float SatYmax = 2.;

const float LumYmin = 0.;
const float LumYmax = 2.;


float vec4ToFloat(vec4 val)
{
    // Combine components with different precision scaling
    float res = val.x * 255. + val.y + val.z / 255.;
    // Flip sign based on w component (0.5 threshold)
    res = val.w < 0.5 ? res : -res;
    return res;
}


float getCurveParam(sampler2D curveTex, float x, float minVal, float maxVal)
{
    float val = vec4ToFloat(texture2D(curveTex, vec2(x, 0.5)));
    // Linearly maps a normalized [0,1] value to [minVal, maxVal] range
    val = minVal + val * (maxVal - minVal);
    return val;
}


float getCurveSlope(sampler2D curveTex, float x, float minVal, float maxVal)
{
    float xLeft = max(x - StepX, 0.);
    float xRight = min(x + StepX, 0.);
    float pLeft = getCurveParam(curveTex, xLeft, minVal, maxVal);
    float pRight = getCurveParam(curveTex, xRight, minVal, maxVal);
    float slope = (pRight - pLeft) / (xRight - xLeft);
    slope = slope / (maxVal - minVal);  // to get normalized slope
    return slope;
}


// hue = [0, 360]
float getHue(vec3 rgb, float cmin, float cmax, float delta)
{
    float r = rgb.r;
    float g = rgb.g;
    float b = rgb.b;
    // Following code is to compute hue by hsl formula
    float h = 0.;
    if (delta == 0.) {
        h = 0.; // Achromatic case
    } else {
        if (cmax == r) {
            h = 60. * ((g - b) / delta);   // [-60, 60]
            if (h < 0.) {
                h += 360.; // shift to [0, 360]
            }
        } else if (cmax == g) {
            h = 60. * ((b - r) / delta) + 120.; // (60, 180]
        } else { // cmax == b
            h = 60. * ((r - g) / delta) + 240.; // (180, 300)
        }
    }
    return h;
}


/**
 * 色相调节本质上是把颜色从RGB空间转换到YUV空间，然后旋转UV来实现，
 * 由于RGB转YUV，UV旋转，以及YUV转RGB都可以通过3x3矩阵来实现，所以最终可以合并成一个矩阵。
 */
vec3 adjustHue(vec3 rgb, float hueParam)
{
    float wr = LuminanceWeight.r;
    float wg = LuminanceWeight.g;
    float wb = LuminanceWeight.b;
    float wrInv = 1. - wr;
    float wgInv = 1. - wg;
    float wbInv = 1. - wb;
    float angle = radians(-hueParam);
    float cosA = cos(angle);
    float sinA = sin(angle);

    float rr = wr + wrInv * cosA - wr * sinA;
    float rg = wg - wg * cosA - wg * sinA;
    float rb = wb - wb * cosA + wbInv * sinA;
    float gr = wr - wr * cosA + 0.1427 * sinA;
    float gg = wg + wgInv * cosA + 0.1404 * sinA;
    float gb = wb - wb * cosA - 0.2831 * sinA;
    float br = wr - wr * cosA - wrInv * sinA;
    float bg = wg - wg * cosA + wg * sinA;
    float bb = wb + wbInv * cosA + wb * sinA;

    /*
     * res.r = dot(rgb, vec3(rr, rg, rb));
     * res.g = dot(rgb, vec3(gr, gg, gb));
     * res.b = dot(rgb, vec3(br, bg, bb));
     */
    
    mat3 hueMatrix = mat3(
        rr, rg, rb,
        gr, gg, gb,
        br, bg, bb
    );
    vec3 res = rgb * hueMatrix;

    return res;
}


vec3 adjustSaturation(vec3 rgb, float satParam)
{
    vec3 resColor = rgb;
    vec3 baseAdjustVec = LuminanceWeight * (1. - satParam);

    vec3 adjustVec = baseAdjustVec + vec3(satParam, 0., 0.);
    resColor.r = dot(rgb, adjustVec);

    adjustVec = baseAdjustVec + vec3(0., satParam, 0.);
    resColor.g = dot(rgb, adjustVec);

    adjustVec = baseAdjustVec + vec3(0., 0., satParam);
    resColor.b = dot(rgb, adjustVec);

    return resColor;
}


vec3 adjustLuminance(vec3 rgb, float lumParam)
{
    float oriLum = dot(rgb, LuminanceWeight);
    float diff = oriLum * (lumParam - 1.);
    vec3 res = rgb + diff;
    return res;
}


/**
 * 对比发现，当rgb三个数值的中位数与cmin，cmax距离都较远时，需要做一定的尺度修正; 另外修正因子跟曲线斜率也有一定关系，曲线越平就越不用修正。
 */
float getSat2LumScale(vec3 rgb, float cmin, float cmax, float slope)
{
    if (cmax == cmin) {
        return 1.;
    }

    float cmid = rgb.r + rgb.g + rgb.b - cmax - cmin;
    slope = clamp(slope, -1., 1.);
    float s = abs(slope * (cmax - cmid) * (cmid - cmin) / (cmax - cmin));
    s = clamp(s, 0., 1.);
    float scale = 1. - s;
    if (sign(slope) < 0.) {
        scale = 1. / scale;
    }
    return scale;
}


float getHue2LumScale(float sat)
{
    float scale = 2.5 * sat;
    return clamp(scale, 0., 1.);
}


/**
 * 调节理论：
 * 色相：色相是颜色的基本属性，表示颜色在光谱中的位置（如红、黄、绿等），在HSL颜色空间中，色相由0到360度的角度表示。
 *     此处图片本身色相数值的计算采用了HSL的公式，但是色相调节使用的是矩阵法，即在YUV空间中旋转UV来实现。
 * 饱和度：饱和度是颜色的纯度或鲜艳程度，此处饱和度的衡量使用rgb的最大值与最小值的差值。
 *     饱和度参数的默认值是1，表示不做饱和度调节，当参数为p时，会把原饱和度s调节为p*s，所以不同饱和度曲线之间耦合采用乘法。
 * 亮度：亮度是颜色的明暗程度，此处亮度由BT.709公式对RGB进行加权求和得到。
 *     亮度参数的耦合同样使用乘法。在靠近灰度色彩时，色相对色彩的贡献度较低，所以对亮度的影响也需降低。
 */
void main()
{
    vec4 oriColor = texture2D(u_inputTexture, uv0);
    float cmin = min(oriColor.r, min(oriColor.g, oriColor.b));
    float cmax = max(oriColor.r, max(oriColor.g, oriColor.b));
    float sat = clamp(cmax - cmin, 0., 1.);  // delta is treated as saturation
    float lum = dot(oriColor.rgb, LuminanceWeight);
    float hue = getHue(oriColor.rgb, cmin, cmax, sat) / HueRange;

    // Get the curve parameters from the texture
    float hue2hueParam = getCurveParam(u_curveHue2hue, hue, HueYmin, HueYmax);
    float hue2satParam = getCurveParam(u_curveHue2sat, hue, SatYmin, SatYmax);
    float hue2lumParam = getCurveParam(u_curveHue2lum, hue, LumYmin, LumYmax);
    float lum2satParam = getCurveParam(u_curveLum2sat, lum, SatYmin, SatYmax);
    float sat2satParam = getCurveParam(u_curveSat2sat, sat, SatYmin, SatYmax);
    float sat2lumParam = getCurveParam(u_curveSat2lum, sat, LumYmin, LumYmax);

    // refinement for Sat2Lum
    float s = 1.;
    float sat2lumSlope = getCurveSlope(u_curveSat2lum, sat, LumYmin, LumYmax);
    s = getSat2LumScale(oriColor.rgb, cmin, cmax, sat2lumSlope);
    sat2lumParam = sat2lumParam * s;

    // refinement for Hue2Lum
    s = getHue2LumScale(sat);
    hue2lumParam = hue2lumParam * s + 1. - s;   // mix(1, hue2lumP, s)

    // Process parameter coupling
    float satParam = hue2satParam * lum2satParam * sat2satParam;
    float lumParam = hue2lumParam * sat2lumParam;

    // Adjust the color
    vec3 resColor = oriColor.rgb;
    resColor = adjustSaturation(resColor, satParam);
    resColor = adjustLuminance(resColor, lumParam);
    resColor = adjustHue(resColor, hue2hueParam);

    gl_FragColor = vec4(resColor, oriColor.a);
}
