---@class SeekModeScript: ScriptComponent

local exports = exports or {}
local SeekModeScript = SeekModeScript or {}
SeekModeScript.__index = SeekModeScript

------------ util functions ------------
local function parsePoints(coords, is_padding)
    local ptsNum = coords:size() / 2
    local anchorPtsNum = coords:size() / 6
    local pts = Amaz.Vec2Vector()

    if is_padding == true and anchorPtsNum > 0 then
        -- Pad both sides with triplets: Left pad uses last triplet, right pad uses first triplet
        -- Left padding
        for k = 3, 1, -1 do
            local i = ptsNum - k
            pts:pushBack(Amaz.Vector2f(coords:get(i * 2) - 1, coords:get(i * 2 + 1))) -- Shift x coordinate to <0 range
        end
        -- Original points
        for i = 0, ptsNum - 1 do
            pts:pushBack(Amaz.Vector2f(coords:get(i * 2), coords:get(i * 2 + 1)))
        end
        -- Right padding
        for i = 0, 2 do
            pts:pushBack(Amaz.Vector2f(coords:get(i * 2) + 1, coords:get(i * 2 + 1))) -- Shift x coordinate to >1 range
        end
    else
        for i = 0, ptsNum - 1 do
            pts:pushBack(Amaz.Vector2f(coords:get(i * 2), coords:get(i * 2 + 1)))
        end
    end
    anchorPtsNum = pts:size() / 3
    return anchorPtsNum, pts
end

------------ class functions for ScriptComponent ------------
function SeekModeScript.new(construct, ...)
    local self = setmetatable({}, SeekModeScript)

    if construct and SeekModeScript.constructor then SeekModeScript.constructor(self, ...) end

    -- ​Anchor Points​​: The curve must pass through these points (start and end).​
    -- Control Points​​: Influence the curve’s shape but are not interpolated (the curve does not necessarily pass through them).

    -- default points (a hozirontal line along middle of y-axis)
    self.defaultEmptyPts = Amaz.Vec2Vector()
    self.defaultPts = Amaz.Vec2Vector()
    self.defaultPts:pushBack(Amaz.Vector2f(0., 0.5))
    self.defaultPts:pushBack(Amaz.Vector2f(0., 0.5))
    self.defaultPts:pushBack(Amaz.Vector2f(0.5, 0.5))
    self.defaultPts:pushBack(Amaz.Vector2f(0.5, 0.5))
    self.defaultPts:pushBack(Amaz.Vector2f(1., 0.5))
    self.defaultPts:pushBack(Amaz.Vector2f(1., 0.5))
    return self
end


function SeekModeScript:reset()
    self.matHue2hue:setVec2Vector("u_points", self.defaultEmptyPts)
    self.matHue2lum:setVec2Vector("u_points", self.defaultEmptyPts)
    self.matHue2sat:setVec2Vector("u_points", self.defaultEmptyPts)
    self.matLum2sat:setVec2Vector("u_points", self.defaultPts)
    self.matSat2sat:setVec2Vector("u_points", self.defaultPts)
    self.matSat2lum:setVec2Vector("u_points", self.defaultPts)

    self.matHue2hue:setInt("u_anchorPtsNum", 0)
    self.matHue2lum:setInt("u_anchorPtsNum", 0)
    self.matHue2sat:setInt("u_anchorPtsNum", 0)
    self.matLum2sat:setInt("u_anchorPtsNum", 2)
    self.matSat2sat:setInt("u_anchorPtsNum", 2)
    self.matSat2lum:setInt("u_anchorPtsNum", 2)
end


function SeekModeScript:onStart(comp)
    self.matHue2hue = comp.entity:searchEntity("Entity_Hue_vs_Hue"):getComponent("MeshRenderer").material
    self.matHue2lum = comp.entity:searchEntity("Entity_Hue_vs_Lum"):getComponent("MeshRenderer").material
    self.matHue2sat = comp.entity:searchEntity("Entity_Hue_vs_Sat"):getComponent("MeshRenderer").material
    self.matLum2sat = comp.entity:searchEntity("Entity_Lum_vs_Sat"):getComponent("MeshRenderer").material
    self.matSat2sat = comp.entity:searchEntity("Entity_Sat_vs_Sat"):getComponent("MeshRenderer").material
    self.matSat2lum = comp.entity:searchEntity("Entity_Sat_vs_Lum"):getComponent("MeshRenderer").material
    self.matAdjust = comp.entity:searchEntity("Entity_Adjust"):getComponent("MeshRenderer").material

    self:reset()
end


function SeekModeScript:seekToTime(comp, time)
    if self.first == nil then
        self.first = true
        self:onStart(comp)
    end
end


function SeekModeScript:onUpdate(comp, detalTime)
    self:seekToTime(comp, detalTime)
end


function SeekModeScript:onEvent(sys, event)
    if self.first == nil then
        self.first = true
        self:onStart(sys)
    end

    if event.type == Amaz.AppEventType.SetEffectIntensity then
        if ("hue_vs_hue_points" == event.args:get(0)) then
            local anchorPtsNum, pts = parsePoints(event.args:get(1), true)
            self.matHue2hue:setInt("u_anchorPtsNum", anchorPtsNum)
            self.matHue2hue:setVec2Vector("u_points", pts)
        elseif ("hue_vs_lum_points" == event.args:get(0)) then
            local anchorPtsNum, pts = parsePoints(event.args:get(1), true)
            self.matHue2lum:setInt("u_anchorPtsNum", anchorPtsNum)
            self.matHue2lum:setVec2Vector("u_points", pts)
        elseif ("hue_vs_sat_points" == event.args:get(0)) then
            local anchorPtsNum, pts = parsePoints(event.args:get(1), true)
            self.matHue2sat:setInt("u_anchorPtsNum", anchorPtsNum)
            self.matHue2sat:setVec2Vector("u_points", pts)
        elseif ("lum_vs_sat_points" == event.args:get(0)) then
            local anchorPtsNum, pts = parsePoints(event.args:get(1), false)
            self.matLum2sat:setInt("u_anchorPtsNum", anchorPtsNum)
            self.matLum2sat:setVec2Vector("u_points", pts)
        elseif ("sat_vs_sat_points" == event.args:get(0)) then
            local anchorPtsNum, pts = parsePoints(event.args:get(1), false)
            self.matSat2sat:setInt("u_anchorPtsNum", anchorPtsNum)
            self.matSat2sat:setVec2Vector("u_points", pts)
        elseif ("sat_vs_lum_points" == event.args:get(0)) then
            local anchorPtsNum, pts = parsePoints(event.args:get(1), false)
            self.matSat2lum:setInt("u_anchorPtsNum", anchorPtsNum)
            self.matSat2lum:setVec2Vector("u_points", pts)
        end

        if event.args:size() == 2 and event.args:get(0) == "reset_params" and event.args:get(1) == 1 then
            self:reset()
        end
    end
end


exports.SeekModeScript = SeekModeScript
return exports
